--------------------------------------------------------------------------------
-- File: top_level.vhd
-- File history:

-- Description: 
--
-- A test using the SDRAM controller. Instantiating it in VHDL to test read writes.
-- Writing 16 lots of 32 bit values, and then doing 16 reads, manually comparing the
--values being returned.
--
-- Author: Dan Potter
--
--------------------------------------------------------------------------------


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
library STD;

entity top_level is

	port (
    
    i_sysclk : in std_logic;
    i_sdrclk : in std_logic;
    i_tACclk : in std_logic;
    i_sw2 : in std_logic;
    i_sw3 : in std_logic;
    o_led : out std_logic_vector(7 downto 0);
    i_pll_locked : in std_logic;
    o_pll_reset : out std_logic;
    o_sdr_CKE : out std_logic;
    o_sdr_n_CS : out std_logic;
    o_sdr_n_WE : out std_logic;
    o_sdr_n_RAS : out std_logic;
    o_sdr_n_CAS : out std_logic;
    o_sdr_BA : out std_logic_vector(1 downto 0);
    o_sdr_ADDR : out std_logic_vector(12 downto 0);
    i_sdr_DATA : in std_logic_vector(15 downto 0);
    o_sdr_DATA : out std_logic_vector(15 downto 0);
    o_sdr_DATA_oe : out std_logic_vector(15 downto 0);
    o_sdr_DQM : out std_logic_vector(1 downto 0) 
    
    );
end top_level;

architecture architecture_top_level of top_level is

COMPONENT sdram_controller is
PORT (
    i_we : in std_logic;
    i_sysclk : in std_logic;
    i_arst : in std_logic;
    i_sdrclk : in std_logic;
    i_tACclk : in std_logic;
    i_pll_locked : in std_logic;
    i_re : in std_logic;
    i_last : in std_logic;
    o_dbg_tRTW_done : out std_logic;
    o_dbg_ref_req : out std_logic;
    o_dbg_wr_ack : out std_logic;
    o_dbg_rd_ack : out std_logic;
    o_dbg_n_CS : out std_logic_vector(1 downto 0);
    o_dbg_n_RAS : out std_logic_vector(1 downto 0);
    o_dbg_n_CAS : out std_logic_vector(1 downto 0);
    o_dbg_n_WE : out std_logic_vector(1 downto 0);
    o_dbg_BA : out std_logic_vector(3 downto 0);
    o_dbg_ADDR : out std_logic_vector(25 downto 0);
    o_dbg_DATA_out : out std_logic_vector(15 downto 0);
    o_dbg_DATA_in : out std_logic_vector(15 downto 0);
    i_addr : in std_logic_vector(24 downto 0);
    i_din : in std_logic_vector(31 downto 0);
    i_dm : in std_logic_vector(3 downto 0);
    o_dout : out std_logic_vector(31 downto 0);
    o_sdr_state : out std_logic_vector(3 downto 0);
    o_sdr_init_done : out std_logic;
    o_wr_ack : out std_logic;
    o_rd_ack : out std_logic;
    o_ref_req : out std_logic;
    o_rd_valid : out std_logic;
    o_sdr_CKE : out std_logic_vector(1 downto 0);
    o_sdr_n_CS : out std_logic_vector(1 downto 0);
    o_sdr_n_RAS : out std_logic_vector(1 downto 0);
    o_sdr_n_CAS : out std_logic_vector(1 downto 0);
    o_sdr_n_WE : out std_logic_vector(1 downto 0);
    o_sdr_BA : out std_logic_vector(3 downto 0);
    o_sdr_ADDR : out std_logic_vector(25 downto 0);
    o_sdr_DATA : out std_logic_vector(31 downto 0);
    o_sdr_DATA_oe : out std_logic_vector(31 downto 0);
    i_sdr_DATA : in std_logic_vector(31 downto 0);
    o_sdr_DQM : out std_logic_vector(3 downto 0);
    o_dbg_dly_cnt_b : out std_logic_vector(5 downto 0);
    o_dbg_tRCD_done : out std_logic
);
END COMPONENT;

    type t_state is (wait_init, wait_wr_ack, state_wr, state_wr2, state_wr_done, state_rd, state_rd_done);
    
    signal state : t_state;
 
type t_write_array is array (0 to 15) of std_logic_vector(31 downto 0);

-- Try assigning values in one big block, 16 lots of 32 bit values, 
constant c16x32_write_array : t_write_array := (x"00001111", x"22223333",x"44445555", x"66667777",
                                                x"88889999", x"AAAABBBB",x"CCCCDDDD", x"EEEEFFFF",
                                                x"0123FFFF", x"1234EEEE",x"2345DDDD", x"3456CCCC",
                                                x"4567BBBB", x"5678AAAA",x"67899999", x"789A8888");
                                                
signal s_we : std_logic;
signal s_wr_ack : std_logic;
signal s25_addr : std_logic_vector(24 downto 0);
signal s32_din : std_logic_vector(31 downto 0);
signal s32_dout : std_logic_vector(31 downto 0);
signal s4_dm : std_logic_vector(3 downto 0) := x"0";
signal s_last : std_logic;
signal s_re : std_logic;
signal s_rd_ack : std_logic;
signal s_rd_valid : std_logic;
signal s_sdr_init_done : std_logic;

signal  sint_cycle_count : integer range 0 to 20 := 0;

signal  s26_sdr_ADDR : std_logic_vector(25 downto 0); 
signal  s32_out_sdr_DATA : std_logic_vector(31 downto 0); 
signal  s32_out_sdr_DATA_oe : std_logic_vector(31 downto 0);
signal  s32_in_sdr_DATA : std_logic_vector(31 downto 0);
signal  s2_sdr_CKE : std_logic_vector(1 downto 0);
signal  s2_sdr_n_CS : std_logic_vector(1 downto 0);
signal  s2_sdr_n_RAS : std_logic_vector(1 downto 0);
signal  s2_sdr_n_CAS : std_logic_vector(1 downto 0);
signal  s2_sdr_n_WE : std_logic_vector(1 downto 0);  
signal  s4_sdr_BA : std_logic_vector(3 downto 0);  
signal  s4_sdr_DQM  : std_logic_vector(3 downto 0);  
signal  s4_sdr_state : std_logic_vector(3 downto 0);  

signal s_reset : std_logic;

begin

s_reset <= not i_pll_locked;

s32_in_sdr_DATA <= x"0000" & i_sdr_DATA;

o_sdr_CKE <= s2_sdr_CKE(0);
o_sdr_n_CS <= s2_sdr_n_CS(0);
o_sdr_n_RAS <= s2_sdr_n_RAS(0);
o_sdr_n_CAS <= s2_sdr_n_CAS(0);
o_sdr_n_WE <= s2_sdr_n_WE(0); 
o_sdr_BA <= s4_sdr_BA(1 downto 0);
o_sdr_ADDR <= s26_sdr_ADDR(12 downto 0);
o_sdr_DATA <= s32_out_sdr_DATA(15 downto 0);
o_sdr_DATA_oe <= s32_out_sdr_DATA_oe(15 downto 0);
o_sdr_DQM <= s4_sdr_DQM(1 downto 0);

o_pll_reset <= i_sw3;

sdram_controller_map : sdram_controller port map (
    i_we => s_we,
    i_sysclk => i_sysclk,
    i_arst => s_reset,
    i_sdrclk => i_sdrclk,
    i_tACclk => i_tACclk,
    i_pll_locked => '1',
    i_re => s_re,
    i_last => s_last,
    o_dbg_tRTW_done => open,
    o_dbg_ref_req => open,
    o_dbg_wr_ack => open,
    o_dbg_rd_ack => open,
    o_dbg_n_CS => open,
    o_dbg_n_RAS => open,
    o_dbg_n_CAS => open,
    o_dbg_n_WE => open,
    o_dbg_BA => open,
    o_dbg_ADDR => open,
    o_dbg_DATA_out => open,
    o_dbg_DATA_in => open,
    i_addr => s25_addr,
    i_din => s32_din,
    i_dm => s4_dm,
    o_dout => s32_dout,
    o_sdr_state => s4_sdr_state,
    o_sdr_init_done => s_sdr_init_done,
    o_wr_ack => s_wr_ack,
    o_rd_ack => s_rd_ack,
    o_ref_req => open,
    o_rd_valid => s_rd_valid,
    o_sdr_CKE => s2_sdr_CKE,
    o_sdr_n_CS => s2_sdr_n_CS,
    o_sdr_n_RAS => s2_sdr_n_RAS,
    o_sdr_n_CAS => s2_sdr_n_CAS,
    o_sdr_n_WE => s2_sdr_n_WE,
    o_sdr_BA => s4_sdr_BA,
    o_sdr_ADDR => s26_sdr_ADDR,
    o_sdr_DATA => s32_out_sdr_DATA,
    o_sdr_DATA_oe => s32_out_sdr_DATA_oe,
    i_sdr_DATA => s32_in_sdr_DATA,
    o_sdr_DQM => s4_sdr_DQM,
    o_dbg_dly_cnt_b => open,
    o_dbg_tRCD_done => open
    
    );


s32_din <= c16x32_write_array(sint_cycle_count);
   
read_write_gen : process (s_reset, i_sysclk)
begin
    if (s_reset = '1') then
        s25_addr <= '0'& x"000000";
        state <= wait_init;
        s_we <= '0';
        s_last <= '0';
        s_re <= '0';
        sint_cycle_count <= 0;
    else
        if rising_edge(i_sysclk) then
            s_last <= '0';
            s_re <= '0';
            s_we <= '0';
            
            case state is
                when wait_init =>
                    if s_sdr_init_done = '1' then
                        state <= state_wr;
                        sint_cycle_count <= 0;
                    end if;
                    
                when state_wr =>
                    s_we <= '1';
                    if (s_wr_ack = '1') then
                        sint_cycle_count <= sint_cycle_count + 1;
                        s25_addr <= s25_addr + 2;
                        if sint_cycle_count = 14 then
                            state <= state_wr_done;  
                            s_last <= '1';
                            o_led <= s32_dout(7 downto 0);
                        end if;
                    end if;
                   
                when state_wr_done =>
                    if sint_cycle_count = 2 then
                        sint_cycle_count <= 0;
                        state <= state_rd;  
                        s_re <= '1';
                        s25_addr <= '0'& x"000000";
                    else
                        sint_cycle_count <= sint_cycle_count + 1;
                    end if; 
                    
                when state_rd =>  
                    s_re <= '1';
                    state <= state_rd;
                    if (s_rd_ack = '1') then  
                        s25_addr <= s25_addr + 2;
                        if sint_cycle_count /= 14 then
                            sint_cycle_count <= sint_cycle_count + 1;
                        else
                            state <= state_rd_done;  
                            s_re <= '1';
                            s_last <= '1';
                            s25_addr <= '0'& x"000000";
                            sint_cycle_count <= 0;
                            --Added in some logic for the LED's so the i_sda dosn't get optimised out.
                            o_led(0) <= s32_dout(0) or s32_dout(1);
                            o_led(1) <= s32_dout(2) or s32_dout(3);
                            o_led(2) <= s32_dout(4) or s32_dout(5);
                            o_led(3) <= s32_dout(6) or s32_dout(7);
                            o_led(4) <= s32_dout(8) or s32_dout(9);
                            o_led(5) <= s32_dout(10) or s32_dout(11);
                            o_led(6) <= s32_dout(12) or s32_dout(13);
                            o_led(7) <= s32_dout(14) or s32_dout(15);                          
                         end if;
                    end if;
                    
                when state_rd_done => 
                    s_re <= '0';
                    if (s_rd_valid = '0') then
                        state <= wait_init;  
                    end if;
                    
                when others =>
                    null;
            end case;
        end if;
    end if;
end process;


end architecture_top_level;